package jadean.dean.java.resourceparser.jarparser;


import jadean.dean.java.JavaPackage;
import jadean.dean.java.constants.JavaProjectConstants;
import jadean.dean.java.utilities.FileUtilities;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.jar.JarOutputStream;
import java.util.jar.Manifest;

public class JarResourceUtilities {
	public static InputStream getInputStreamForEntry(String jarRes, String className) throws IOException {
		final String entry = className.replace('.', JavaProjectConstants.JAR_FILE_ENTRIES_SEPARATOR) + ".class";
		JarFile jf = new JarFile(jarRes);
		Enumeration<JarEntry> e = jf.entries();
		while (e.hasMoreElements()) {
			JarEntry je = e.nextElement();
			if (je.getName().equals(entry)) {
				InputStream is = jf.getInputStream(je);
				//jf.close();
				return is;
			}
		}
		jf.close();
		return null;
	}

	public static List<String> listFiles(String jarRes, JarEntryFilter jefi, JarEntryFormatter jeft) throws IOException {
		JarFile jf = new JarFile(jarRes);
		Enumeration<JarEntry> e = jf.entries();
		ArrayList<String> a = new ArrayList<String>();
		while (e.hasMoreElements()) {
			JarEntry je = e.nextElement();
			if (jefi.accept(je)) {
				a.add(jeft.format(je));
			}
		}
		jf.close();
		return a;
	}

	public static HashMap<JavaPackage, List<String>> listFiles(List<JavaPackage> jarRes, JarEntryFilter jefi, JarEntryFormatter jeft) throws IOException {
		HashMap<JavaPackage, List<String>> ret = new HashMap<JavaPackage, List<String>>();
		for (JavaPackage p: jarRes) {
			ret.put(p, listFiles(p.getPath(), jefi, jeft));
		}
		return ret;
	}

	public static void removeEntries(String jarFileName, ArrayList<String> entriesToDelete) throws IOException {
		String sourceJarFileName = jarFileName;
		String destinationJarFileName = jarFileName + ".tmp";
		
		copyUndeletedEntries(entriesToDelete, sourceJarFileName, destinationJarFileName);
		
		FileUtilities.renameTo(destinationJarFileName, sourceJarFileName);
	}

	public static void deleteEmptyFolders(String jarFileName) throws IOException {
		// TODO Auto-generated method stub
		JarFile jarFile = new JarFile(jarFileName);
		Enumeration<JarEntry> e = jarFile.entries();
		Set<String> dirEntries = new HashSet<String>();
		Set<String> fileEntries = new HashSet<String>();
		while (e.hasMoreElements()) {
			JarEntry je = e.nextElement();
			if (je.isDirectory()) {
				dirEntries.add(je.getName());
			}
			else {
				fileEntries.add(je.getName());
			}
		}
		jarFile.close();
		
		/**
		 * 1. zober adresar
		 * 2. ak je prazdny, vymaz ho
		 * 3. ak nie je prazdny, pozri sa ci obsahuje subory
		 * 4. ak obsahuje subory, nechaj ho
		 * 5. ak neobsahuje subory, zisti ci su adresare vymazane, ak su, pridaj ho a goto 1.
		 */
		
		Set<String> entriesToDelete = new HashSet<String>();
		ArrayList<String> sortedDirEntries = new ArrayList<String>(dirEntries);
		Collections.sort(sortedDirEntries, new Comparator<String>() {
			public int compare(String arg0, String arg1) {
				if (arg0.length() < arg1.length()) {
					return -1;
				}
				else if (arg0.length() > arg1.length()) {
					return 1;
				}
				return 0;
			}
		});
		for (String dir: sortedDirEntries) {
			if (!entriesToDelete.contains(dir)) {
				deleteDir(dir, dirEntries, fileEntries, entriesToDelete);
			}
		}
		
		removeEntries(jarFileName, new ArrayList<String>(entriesToDelete));
	}
	
	private static void deleteDir(String dir, Set<String> dirs, Set<String> files, Set<String> deletedDirs) {
		if (isEmptyDir(dir, dirs, files)) {
			deletedDirs.add(dir);
		}
		else {
			if (!containsDirFiles(dir, files)) {
				for (String _dir: getDirsInDir(dir, dirs)) {
					deleteDir(_dir, dirs, files, deletedDirs);
				}				
				boolean allDirsDeleted = true;
				for (String _dir: getDirsInDir(dir, dirs)) {
					allDirsDeleted &= deletedDirs.contains(_dir);
				}
				if (allDirsDeleted) {
					deletedDirs.add(dir);
				}
			}
		}
	}
	
	private static boolean isEmptyDir(String dir, Set<String> dirs, Set<String> files) {
		boolean isEmpty = true;
		
		for (String _dir: dirs) {
			if (_dir.startsWith(dir) && !_dir.equals(dir)) {
				isEmpty = false;
			}
		}
		
		for (String _file: files) {
			if (_file.startsWith(dir)) {
				isEmpty = false;
			}
		}
		
		return isEmpty;
	}
	
	private static boolean containsDirFiles(String dir, Set<String> files) {
		boolean isEmpty = false;
		
		for (String _file: files) {
			if (_file.startsWith(dir)) {
				isEmpty = true;
			}
		}
		
		return isEmpty;		
	}
	
	private static Set<String> getDirsInDir(String dir, Set<String> dirs) {
		Set<String> ret = new HashSet<String>();
		
		for (String _dir: dirs) {
			if (_dir.startsWith(dir) && !_dir.equals(dir) && _dir.replace(dir, "").split("/").length == 2) {
				ret.add(_dir);
			}
		}
		
		return ret;
	}	
	
	public static boolean isEntryInList(JarEntry je, ArrayList<String> entriesToDelete) {
		for (String s: entriesToDelete) {
			if (je.getName().equals(s) ||
				je.getName().matches(s.replace('.', '/').replace("\\", "\\\\") + "(\\$.+)*\\.class")) {
				return true;
			}
		}		
		return false;
	}

	public static void copyUndeletedEntries(ArrayList<String> entriesToDelete, String sourceJarFileName, String destinationJarFileName) throws IOException {
		JarFile sourceJarFile = new JarFile(sourceJarFileName);
		JarOutputStream destinationJarFile = new JarOutputStream(new FileOutputStream(destinationJarFileName));
		
		Enumeration<JarEntry> e = sourceJarFile.entries();
		while (e.hasMoreElements()) {
			JarEntry je = e.nextElement();
			byte[] buffer = new byte[1024];
			int bytesRead;
			boolean delete = isEntryInList(je, entriesToDelete);

			if (!delete) {
				InputStream entryStream = sourceJarFile.getInputStream(je);
				destinationJarFile.putNextEntry(je);
				while ((bytesRead = entryStream.read(buffer)) != -1) {
					destinationJarFile.write(buffer, 0, bytesRead);
				}
				entryStream.close();
			}
		}
		
		sourceJarFile.close();
		destinationJarFile.close();	
	}

	public static List<String> getClassPathPackages(String jarFileName) {
		if (!hasClassPathPackages(jarFileName)) {
			return new ArrayList<String>();
		}
		JarFile jarFile = null;
		String absolutePath = new File(jarFileName).getAbsolutePath();
		String jarPath = absolutePath.substring(0, absolutePath.lastIndexOf(File.separatorChar));
		ArrayList<String> classPathPackages = new ArrayList<String>();
		try {
			jarFile = new JarFile(jarFileName); 
			Manifest manifest = jarFile.getManifest();
			String classPathAttribute = manifest.getMainAttributes().getValue("Class-Path");

			StringTokenizer tokenizer = new StringTokenizer(classPathAttribute);
			while (tokenizer.hasMoreElements()) {
				String token = tokenizer.nextToken();				
				classPathPackages.add(jarPath + File.separatorChar + token.replace('/', File.separatorChar));
			}

			return classPathPackages;
		}
		catch (IOException e) {
			return new ArrayList<String>();
		}
		finally {
			try {
				jarFile.close();
			} catch (Exception e) {
			}
		}
	}

	public static boolean hasClassPathPackages(String jarFileName) {
		JarFile jarFile = null;
		try {
			jarFile = new JarFile(jarFileName); 
			Manifest manifest = jarFile.getManifest();
			String classPathAttribute = manifest.getMainAttributes().getValue("Class-Path");
			return classPathAttribute != null;
		}
		catch (IOException e) {
			return false;
		}
		finally {
			try {
				jarFile.close();
			} catch (Exception e) {
			}
		}		
	}
}
